<?php
/**
 * @package     FILEman
 * @copyright   Copyright (C) 2011 Timble CVBA. (http://www.timble.net)
 * @license     GNU GPLv3 <http://www.gnu.org/licenses/gpl.html>
 * @link        http://www.joomlatools.com
 */

class ComFilemanJobScans extends ComSchedulerJobAbstract
{
    const STATUS_PENDING = 0;

    const STATUS_SENT = 1;

    const STATUS_FAILED = 2;

    const STATUS_DEFERRED = 3;

    const STATUS_ABANDONED = 4;

    const MAXIMUM_PENDING_SCANS = 6;

    protected $_scans_model;

    /**
     * The number of retries per scan before abandoning
     *
     * @var int
     */
    protected $_scan_retries_limit = 3;

    /**
     * The number of halt retries (all scans) before halting the job
     *
     * @var int
     */
    protected $_halt_retries_limit = 5;

    /**
     * The number of failed scans for triggering a halt
     *
     * @var int
     */
    protected $_halt_fail_trigger = 5;

    public function __construct(KObjectConfig $config)
    {
        parent::__construct($config);

        $this->_scans_model = $config->scans_model;
    }

    protected function _initialize(KObjectConfig $config)
    {
        $config->append(array(
            'scans_model' => 'com://admin/fileman.model.scans',
            'frequency'   => ComSchedulerJobInterface::FREQUENCY_EVERY_FIVE_MINUTES
        ));

        parent::_initialize($config);
    }

    public function run(ComSchedulerJobContextInterface $context)
    {
        try {
            $i = 0;

            if (!$this->_isHalted($context))
            {
                while ($context->hasTimeLeft() && $i < 4)
                {
                    $this->purgeStaleScans();

                    if (!$scan = $context->scan)
                    {
                        $this->_handleFailedScans($context);

                        $scan = $this->_getScansModel()
                                     ->status(self::STATUS_PENDING)
                                     ->limit(1)
                                     ->sort('created_on')->direction('desc')
                                     ->fetch();
                    }
                    elseif ($scan->status == self::STATUS_FAILED)
                    {
                            $this->_processFailedScan($scan, $context); // Handle failed scan
                    }

                    if (!$scan->isNew())
                    {
                        if (!in_array($scan->status, array(self::STATUS_FAILED, self::STATUS_ABANDONED, self::STATUS_SENT, self::STATUS_DEFERRED)))
                        {
                            $this->sendScan($scan, $context);

                            if ($scan->status == self::STATUS_SENT) {
                                $context->log('Sent request to scan ' . $scan->identifier);
                            }
                        }
                    }

                    $i++;
                }
            }
        }
        catch (Exception $e) {
            $context->log($e->getMessage());
        }

        return $this->complete();
    }

    protected function _isHalted(ComSchedulerJobContextInterface $context)
    {
        $result = true;

        $state = $context->getState();

        if ($state->halt)
        {
            $halted_on = new DateTime($state->halted_on, new DateTimeZone('UTC'));

            // Reset the retry count next month

            if ($halted_on->format('Y') != gmdate('Y', time())) {
                $state->halt_count = 0;
            } elseif ($halted_on->format('n') < gmdate('n', time())) {
                $state->halt_count = 0;
            }

            if ($state->halt_count < $this->_halt_retries_limit)
            {
                $modifier = sprintf('+%d hours', pow(2, $state->halt_count) - 1);

                $halted_on->modify($modifier);

                if ($halted_on->getTimestamp() < time())
                {
                    // Un-halt the job so that scans are sent

                    $state->halt = false;

                    // Reset failed scans

                    $query = $this->getObject('lib:database.query.update');

                    $query->table('fileman_scans')
                          ->values(array('status = :pending'))
                          ->where('status = :failed')
                          ->where('retries < :retries')
                          ->bind(array(
                              'pending' => self::STATUS_PENDING,
                              'failed'  => self::STATUS_FAILED,
                              'retries' => $this->_scan_retries_limit
                          ));

                    $this->getObject('lib:database.adapter.mysqli')->update($query);

                    $context->log(sprintf('FILEman scans job has been un-halted with a halt count of %s', $state->halt_count));

                    $result = false;
                }
            }
        }
        else
        {
            // See if the job should be halted

            $query = $this->getObject('lib:database.query.select');

            $query->table('fileman_scans')
                  ->columns(array('COUNT(*)'))
                  ->where('status = :status')
                  ->bind(array('status' => SELF::STATUS_FAILED));

            $failed = (int) $this->getObject('lib:database.adapter.mysqli')->select($query, KDatabase::FETCH_FIELD);

            if ($failed >= $this->_halt_fail_trigger)
            {
                $state->halt       = true;
                $state->halt_count = $state->halt_count ? $state->halt_count + 1 : 1;
                $state->halted_on  = gmdate('Y-m-d H:i:s', time());

                $context->log(sprintf('FILEman scan jobs has been halted with a halt count of %s', $state->halt_count));
            }
            else $result = false;
        }

        return $result;
    }

    protected function _processFailedScan(KModelEntityInterface $scan, ComSchedulerJobContextInterface $context)
    {
        if ($scan->retries < $this->_scan_retries_limit)
        {
            $parameters = $scan->getParameters();

            $sent_on = $parameters->sent_on ?: $scan->created_on;

            $date = new DateTime($sent_on, new DateTimeZone('UTC'));

            $modifier = sprintf('+%d hours', pow(2, $scan->retries + 1) - 1);

            $date->modify($modifier);

            if ($date->getTimestamp() < time())
            {
                $scan->status  = self::STATUS_PENDING;
                $scan->retries = $scan->retries + 1;

                $context->log(sprintf('FILEman scan with ID %s status has been reset to pending after %s retries', $scan->id, $scan->retries));

            }
        }
        else
        {
            // Set scan as abandoned

            $scan->status = self::STATUS_ABANDONED;
            $scan->save();

            $context->log(sprintf('FILEman scan with ID %s has been abandoned after %s retries', $scan->id, $scan->retries));
        }
    }

    protected function _handleFailedScans(ComSchedulerJobContextInterface $context)
    {
        $query = $this->getObject('lib:database.query.update');

        $query->table('fileman_scans')
              ->values('status = :pending')
              ->bind(array('pending' => self::STATUS_PENDING));

        for ($i = 0; $i < $this->_scan_retries_limit; $i++)
        {
            $condition = sprintf('(retries = :retries_%1$s AND status = :failed AND DATE_ADD(sent_on, INTERVAL %2$s HOUR) < UTC_TIMESTAMP())', $i, pow(2, $i + 1) - 1);

            $query->where($condition, 'OR')->bind(array(
                sprintf('retries_%1$s', $i) => $i
            ));
        }

        $query->bind(array('failed' => self::STATUS_FAILED));

        $adapter = $this->getObject('lib:database.adapter.mysqli');

        if ($result = $adapter->update($query)) {
            $context->log(sprintf('%s failed FILEman scans have been reset to pending', $result));
        }

        $query = $this->getObject('lib:database.query.update');

        $query->table('fileman_scans')->values('status = :abandoned')->where('retries >= :retries')
              ->where('status = :failed')->bind(array(
                'abandoned' => self::STATUS_ABANDONED,
                'retries'   => $this->_scan_retries_limit,
                'failed'    => self::STATUS_FAILED
            ));

        if ($result = $adapter->update($query)) {
            $context->log(sprintf('%s failed FILEman scans have been abandoned', $result));
        }
    }

    public function sendScan($scan, ComSchedulerJobContextInterface $context)
    {
        $entity = $scan->getEntity();

        if (!$entity->isNew())
        {
            if ($scan->status != self::STATUS_SENT)
            {
                if (!$this->isSupported()) {
                    $context->log('Joomlatools Connect is not installed, it is outdated or its credentials are missing');
                }

                if ($this->isLocal()) {
                    $context->log('File scan needs public server');
                }

                if ($this->needsThrottling()) {
                    $context->log('File scan is throttled');
                }

                $parameters = $scan->getParameters();

                $data = array(
                    'download_url' => (string) $this->_getDownloadUrl($entity),
                    'callback_url' => (string) $this->_getCallbackUrl(),
                    'filename'     => Koowa\basename($entity->path),
                    'user_data'    => array(
                        'container' => $entity->container,
                        'folder'    => $entity->folder,
                        'name'      => $entity->name
                    )
                );

                if ($target = $parameters->target) {
                    $data['user_data']['target'] = KObjectConfigJson::unbox($target);
                }

                if ($scan->thumbnail)
                {

                    if ($size = $parameters->size)
                    {
                        $data['thumbnail_size'] = array(
                            'width'  => $size->width,
                            'height' => $size->height
                        );
                    }
                }

                $response = PlgKoowaConnect::sendRequest('scanner/start', ['data' => $data]);

                $scan->status = static::STATUS_SENT;

                if ($response && $response->status_code == 200) {
                    $scan->response = $response->body;
                } else if (!$response || $response->status_code === 401 || $response->status_code === 403) {
                    $scan->status = static::STATUS_FAILED;
                }

                $scan->getParameters()->sent_on = gmdate('Y-m-d H:i:s', time());

                $scan->save();
            }
        }
        else $scan->delete(); // Delete the scan

        return $scan;
    }

    /**
     * Return a callback URL to the plugin with a JWT token
     *
     * @return KHttpUrlInterface
     */
    protected function _getCallbackUrl()
    {
        /** @var KHttpUrlInterface $url */
        $url = clone $this->getObject('request')->getSiteUrl();

        $query = array(
            'option'               => 'com_fileman',
            'view'                 => 'files',
            'format'               => 'json',
            'connect'              => 1,
            //'XDEBUG_SESSION_START' => 'PHPSTORM',
            'token'                => PlgKoowaConnect::generateToken()
        );

        if (substr($url->getPath(), -1) !== '/') {
            $url->setPath($url->getPath().'/');
        }

        $url->setQuery($query);

        return $url;
    }

    /**
     * Return a download URL with a JWT token for the given entity
     *
     * This will bypass all access checks to make sure thumbnail service can access the file
     *
     * @param  KModelEntityInterface $entity
     * @return KHttpUrlInterface
     */
    protected function _getDownloadUrl(KModelEntityInterface $entity)
    {
        /** @var KHttpUrlInterface $url */
        $url       = clone $this->getObject('request')->getSiteUrl();

        $query = array(
            'option'    => 'com_fileman',
            'view'      => 'file',
            'container' => $entity->container,
            'folder'    => $entity->folder,
            'name'      => $entity->name,
            //'XDEBUG_SESSION_START' => 'PHPSTORM',
            'serve'     => 1,
            'connect'   => 1,
            'token'     => PlgKoowaConnect::generateToken()
        );

        $url->setQuery($query);

        return $url;
    }

    public function needsThrottling()
    {
        $count = $this->_getScansModel()->status(self::STATUS_SENT)->count();

        return ($count >= self::MAXIMUM_PENDING_SCANS);
    }


    public function purgeStaleScans()
    {
        /*
         * Set status back to "not sent" for scans that did not receive a response for over an hour
         */
        /** @var KDatabaseQueryUpdate $query */
        $query = $this->getObject('database.query.update');

        $now = gmdate('Y-m-d H:i:s');

        $query
            ->values('status = ' . self::STATUS_PENDING)
            ->table(array('tbl' => 'fileman_scans'))
            ->where('status = ' . self::STATUS_SENT)
            ->where("GREATEST(created_on, modified_on) < DATE_SUB(:now, INTERVAL 5 MINUTE)")
            ->bind(['now' => $now]);

        $this->getObject('com://admin/fileman.database.table.scans')->getAdapter()->update($query);
    }

    public function addScan(KModelEntityInterface $entity, $config)
    {
        $result = false;

        if ($this->isSupported())
        {
            $scan = $this->getScan($entity);

            $scan->setProperties($config)->save();

            if (!$scan->isNew()) {
                $result = $scan;
            }
        }

        return $result;
    }

    public function getScan(KModelEntityInterface $entity)
    {
        $model = $this->_getScansModel();

        $scan = $model->container($entity->container)->folder($entity->folder)->name($entity->name)->fetch();

        if ($scan->isNew())
        {
            $scan = $model->create()->setProperties(array(
                'folder'    => $entity->folder,
                'name'      => $entity->name,
                'container' => $entity->container
            ));
        }

        return $scan;
    }

    public function isSupported()
    {
        return class_exists('PlgKoowaConnect') && PlgKoowaConnect::isSupported()
               && defined('PlgKoowaConnect::VERSION')
               && version_compare(PlgKoowaConnect::VERSION, '2.0.0', '>=');
    }

    public function isLocal()
    {
        return PlgKoowaConnect::isLocal();
    }

    public function isEnabled()
    {
        return $this->isSupported() && !$this->isLocal();
    }

    protected function _getScansModel()
    {
        return $this->getObject($this->_scans_model);
    }
}